# -*- coding: utf-8 -*-
"""
Created on Thu Mar 11 12:45:37 2021

@author: Dyrvik
"""
import numpy as np
import matplotlib.pyplot as plt
import matplotlib.cm as cm
import matplotlib as mpl
import seaborn as sns
import pandas as pd
import os
from mpl_toolkits.axes_grid1 import Divider, Size

def tablelegend(ax, ax_twin, col_labels=None, row_labels=None, title_label="", *args, **kwargs): # edited to work with twinned axis
    """
    Place a table legend on the axes.
    
    Creates a legend where the labels are not directly placed with the artists, 
    but are used as row and column headers, looking like this:
    
    title_label   | col_labels[1] | col_labels[2] | col_labels[3]
    -------------------------------------------------------------
    row_labels[1] |
    row_labels[2] |              <artists go there>
    row_labels[3] |
    
    
    Parameters
    ----------
    
    ax : `matplotlib.axes.Axes`
        The artist that contains the legend table, i.e. current axes instant.
        
    col_labels : list of str, optional
        A list of labels to be used as column headers in the legend table.
        `len(col_labels)` needs to match `ncol`.
        
    row_labels : list of str, optional
        A list of labels to be used as row headers in the legend table.
        `len(row_labels)` needs to match `len(handles) // ncol`.
        
    title_label : str, optional
        Label for the top left corner in the legend table.
        
    ncol : int
        Number of columns.
        

    Other Parameters
    ----------------
    
    Refer to `matplotlib.legend.Legend` for other parameters.
    
    """
    #################### same as `matplotlib.axes.Axes.legend` #####################
    handles, labels, extra_args, kwargs = mpl.legend._parse_legend_args([ax], *args, **kwargs)
    handles_twin, labels_twin, extra_args, kwargs = mpl.legend._parse_legend_args([ax_twin], *args, **kwargs)
    handles = handles + handles_twin
    if len(extra_args):
        raise TypeError('legend only accepts two non-keyword arguments')
    
    if col_labels is None and row_labels is None:
        ax.legend_ = mpl.legend.Legend(ax, handles, labels, **kwargs)
        ax.legend_._remove_method = ax._remove_legend
        return ax.legend_
    #################### modifications for table legend ############################
    else:
        ncol = kwargs.pop('ncol')
        handletextpad = kwargs.pop('handletextpad', 0 if col_labels is None else -2)
        title_label = [title_label]
        
        # blank rectangle handle
        extra = [mpl.patches.Rectangle((0, 0), 1, 1, fc="w", fill=False, edgecolor='none', linewidth=0)]
        
        # empty label
        empty = [""]

        # number of rows infered from number of handles and desired number of columns
        nrow = len(handles) // ncol
        
        # organise the list of handles and labels for table construction
        if col_labels is None:
            assert nrow == len(row_labels), "nrow = len(handles) // ncol = %s, but should be equal to len(row_labels) = %s." % (nrow, len(row_labels))
            leg_handles = extra * nrow
            leg_labels  = row_labels
        elif row_labels is None:
            assert ncol == len(col_labels), "ncol = %s, but should be equal to len(col_labels) = %s." % (ncol, len(col_labels))
            leg_handles = []
            leg_labels  = []
        else:
            assert nrow == len(row_labels), "nrow = len(handles) // ncol = %s, but should be equal to len(row_labels) = %s." % (nrow, len(row_labels))
            assert ncol == len(col_labels), "ncol = %s, but should be equal to len(col_labels) = %s." % (ncol, len(col_labels))
            leg_handles = extra + extra * nrow
            leg_labels  = title_label + row_labels
        for col in range(ncol):
            if col_labels is not None:
                leg_handles += extra
                leg_labels  += [col_labels[col]]
            leg_handles += handles[col*nrow:(col+1)*nrow]
            leg_labels  += empty * nrow
        
        # Create legend
        ax.legend_ = mpl.legend.Legend(ax, leg_handles, leg_labels, ncol=ncol+int(row_labels is not None), handletextpad=handletextpad, **kwargs)
        ax.legend_._remove_method = ax._remove_legend
        return ax.legend_

"""Colourmaps"""
CM_SUBSECTION_GREENS = np.linspace(0.4, 1.0, 1000) 
COLOURS_GREENS = [ cm.Greens(x) for x in CM_SUBSECTION_GREENS ]

CM_SUBSECTION_ORANGES = np.linspace(0.3, 1.0, 1000) 
COLOURS_ORANGES = [ cm.Oranges(x) for x in CM_SUBSECTION_ORANGES ]

CM_SUBSECTION_BLUES = np.linspace(0.3, 1.0, 1000) 
COLOURS_BLUES = [ cm.Blues(x) for x in CM_SUBSECTION_BLUES ]

"""File handling"""
main_folder_path = os.path.dirname(os.path.abspath(__file__))
main_folder_path = '\\\\?\\' + main_folder_path
data_folder_name = 'clean set low threshold processed data'
data_folder_path = os.path.join(main_folder_path,data_folder_name)
export_folder_path = main_folder_path
# =============================================================================
# export_folder_name = 'Plots'
# export_folder_path = os.path.join(main_folder_path,export_folder_name)
# if not os.path.exists(export_folder_path):
#     os.mkdir(export_folder_path) # create folder if non-existent
# =============================================================================
    
# =============================================================================
# selected_files = ['dyrvik_IGLED19-Batch5_002724_3_1.ilv_processed', \
#                  'dyrvik_IGLED19-Batch5_00272c_6_1.ilv_processed', \
#                  'dyrvik_IGLED19-Batch5_002734_6_1.ilv_processed']
# =============================================================================
#272b p6, 272c p6, 2723 p3, 2724 p3, 2727 p3_5, 2728 p5, 272F p5. 2730 p6, 2733 p5, 2734 p6
# 24, 28 (27 similar), 2c (2b similar but more conductive and less shunt like), 30 (2f very similar), 34 (33 similar)

# best of each eqe
selected_files = ['dyrvik_IGLED19-Batch5_002723_3_8.ilv_processed', \
                 'dyrvik_IGLED19-Batch5_00272C_8_5.ilv_processed', \
                 'dyrvik_IGLED19-batch5_002734_5_14.ilv_processed']


"""Some constants for graphical settings"""
lw = 3
ticksize_major = 8
ticksize_minor = 8  

"""Prepare plots"""
label_strings = ['0 cycles',\
                 '10 cycles',\
                 '20 cycles']
    
label_strings_twin = ['0',\
                 '10',\
                 '20']
                                              
colour_array =  np.linspace(0, 999, len(selected_files))

fig = plt.figure(figsize=(7.5, 7.5))
h = [Size.Fixed(1.0), Size.Scaled(1.), Size.Fixed(.2)]
v = [Size.Fixed(0.7), Size.Scaled(1.), Size.Fixed(.5)]
divider = Divider(fig, (0, 0, 1, 1), h, v, aspect=False)
# The width and height of the rectangle are ignored.
ax = plt.axes([0.15, 0.15, 0.85, 0.85])
ax_twin = ax.twinx()
ax.set_xlabel('Bias [V]')
ax.set_ylabel('Current Density [$\mathregular{mA/cm^2}$]')
ax_twin.set_ylabel('Luminance [$\mathregular{cd/m^2}$]')

"""Plot the data"""
for i in range(len(selected_files)):
    """Load data, turn the sign and remove negatives to allow log-plots"""
    file_path = os.path.join(data_folder_path,selected_files[i])
    df = pd.read_csv(file_path)
    
    ax.semilogy(df['LED voltage [V]'], df['LED current [mA/cm2]'], color=COLOURS_GREENS[np.int(colour_array[i])], label=label_strings[i], zorder=3-i)
    ax_twin.semilogy(df['LED voltage [V]'], df['Luminance [cd/m2]'], marker='s', markersize=3.5*lw, markevery=0.1, color=COLOURS_BLUES[np.int(colour_array[i])], label=label_strings_twin[i], zorder=3-i)

"""Annotation"""

ax.annotate("",
            xy=(0.249, 0.45), xycoords='axes fraction',
            xytext=(0.399, 0.45), textcoords='axes fraction',
            arrowprops=dict(facecolor='black', width=lw),
            )
ax.annotate("",
            xy=(0.761, 0.45), xycoords='axes fraction',
            xytext=(0.611, 0.45), textcoords='axes fraction',
            arrowprops=dict(facecolor='black', width=lw),
            )

"""Axis settings"""
ax.set_xlim(left=2.5)
ax.set_ylim(bottom=2e-3)
ax_twin.set_ylim(bottom=5e-0, top=9e3)
ax.xaxis.set_major_locator(mpl.ticker.MultipleLocator(0.5))
ax.xaxis.set_minor_locator(mpl.ticker.MultipleLocator(0.1))


leg = ax.legend(title='Current density',loc='upper left',frameon = False)
leg_twin = ax_twin.legend(title='Luminance',loc='lower right', frameon = False)

# Right-aligning the legend
vp = leg_twin._legend_box._children[-1]._children[0]
for c in vp._children:
    c._children.reverse()
vp.align="right"

"""Graphical details"""
plt.rcParams['font.size'] = 24
plt.rc('axes', labelsize=30)    # fontsize of the x and y labels
plt.rcParams['axes.linewidth'] = lw
plt.rcParams['lines.linewidth'] = lw*1.5

ax.xaxis.set_tick_params(which='major', size=ticksize_major, width=lw)
ax.xaxis.set_tick_params(which='minor', size=ticksize_minor, width=lw)
ax.yaxis.set_tick_params(which='major', size=ticksize_major, width=lw)
ax.yaxis.set_tick_params(which='minor', size=ticksize_minor, width=lw)
ax_twin.yaxis.set_tick_params(which='major', size=ticksize_major, width=lw)
ax_twin.yaxis.set_tick_params(which='minor', size=ticksize_minor, width=lw)


"""Exporting plots"""
plot_name = 'Plot JVL ALD Cycles'
export_name_pdf = plot_name + '.pdf'
export_name_svg = plot_name + '.svg'
#export_name_eps = plot_name + '.eps'
pdf_path = os.path.join(export_folder_path,export_name_pdf)
svg_path = os.path.join(export_folder_path,export_name_svg)
#eps_path = os.path.join(export_folder_path,export_name_eps)
if len(pdf_path) > 255:
    warning_string = 'Warning: plot file path longer than 255 characters, downstream handling issues may occur. Path: ' + pdf_path
    print(warning_string)
fig.savefig(pdf_path, bbox_inches='tight')
fig.savefig(svg_path, bbox_inches='tight')
#fig.savefig(eps_path, bbox_inches='tight')
plt.close(fig)


